import cv2
import math
import yaml
import time
import numpy as np
import copy
import sys
from scipy.optimize import minimize
from sys import argv
from numpy.linalg import inv
from scipy.optimize import minimize
from multiprocessing import Pool, cpu_count
import re

#Признак отладочной печати
DEBUG_PRINT = True
#Минимальное количество пар точек
MIN_MATCHES = 8
#Максимальное количество пар точек
MAX_MATCHES = 128
#Минимальный вес пары точек
MIN_WEIGHT = 0.03

#Расчет маски пересечения изобрадений слева и справа
def mask_process(image, otn_size = 0.8):

    #Ширина пустой области
    w = int(otn_size * image.shape[1] + 0.5)
    #Вектор области
    m = [0] * w + [1] * (image.shape[1] - w)
    #Общий вектор
    mask0 = np.array(m * image.shape[0], dtype='uint8').reshape(image.shape)

    #Ширина пустой области
    w = int((1-otn_size) * image.shape[1] + 0.5)
    #Вектор области
    m = [1] * w + [0] * (image.shape[1] - w)
    #Общий вектор
    mask1 = np.array(m * image.shape[0], dtype='uint8').reshape(image.shape)

    return (mask0, mask1)

#Фильтрация групп по общему углу сшивки
def filter_points(kp0, kp1, matches, width, treshold=0.9):
          
    #Список всех углов
    lst_angle = []
    #Цикл по всем парам точек
    for match in matches:
        #Пара точек
        m0 = match['match']
        #Пара точек
        pnt0 = kp0[m0.queryIdx]; pnt1 = kp1[m0.trainIdx]
        #Приращение по оси XY
        dx = pnt0[0] - (pnt1[0] + width); dy = pnt0[1] - pnt1[1]
        #Угол между точками, град (0..2pi)
        lst_angle.append(math.atan2(dy,dx)/math.pi*180+180)
        
    #Среднее по медиане
    a0 = np.median(lst_angle)
    
    #Список отфильтроанных групп
    flt_matches = []
    #Цикл по всем парам точек
    for i in np.arange(len(lst_angle)):
        #Разница углов
        da = abs(lst_angle[i] - a0)
        #Нормализация разницы углов
        if da > 180: da = 360 - da
        #Список отфильтроанных групп
        flt_matches.append({'match': matches[i]['match'], 'weight': matches[i]['weight'], 'da': da})
    #Сортировка групп по мере подобия (сначала самые правдоподобные)
    flt_matches = sorted(flt_matches, key = lambda x: x['da'])
    #Список совпадений
    return flt_matches[:int(treshold*len(flt_matches)+0.5)]

#Мера близости между двумя группами точек
def measure_pnts(pnts0, pnts1, weights, treshold=0.9):
      
    #Список мер близости
    measures = [math.sqrt((pnts0[i][0] - pnts1[i][0])**2+(pnts0[i][1] - pnts1[i][1])**2) for i in np.arange(len(pnts0))]
       
    #Среднее по медиане
    m0 = np.median(measures)
    #Список отфильтроанных мер подобия
    flt_measures = []
    #Цикл по всем парам точек
    for i in np.arange(len(measures)):
        #Список отфильтроанных групп
        flt_measures.append({'measure': measures[i], 'dm': max(MIN_WEIGHT, (1 - weights[i])) * abs(measures[i] - m0), 'index': i})
    #Сортировка групп по мере подобия (сначала самые правдоподобные)
    flt_measures = sorted(flt_measures, key = lambda x: x['dm'])
    #Размер фильтрованного вектора
    size = int(treshold*len(flt_measures) + 0.5)
    
    #Итоговая мера подобия
    measures = 0
    #Список фильтрованных мер
    for flt in flt_measures[:size]:
        #Итоговая мера подобия
        measures += flt['measure']
    
    #Список совпадений
    return float(measures/size)

#Построение матрицы камеры
def MakeCameraMatrix(fovX, size):
    f = size[0] / (math.tan(fovX / 2.0 ) * 2.0)
    cx = (size[0] / 2.0)
    cy = (size[1] / 2.0)
    return np.array([f, 0, cx, 0, f, cy, 0, 0, 1], dtype='f').reshape((3,3))

#Репроекция точек с учетом дисторсии
def distortPoints(pnts, K, D):
    pnts_und = cv2.undistortPoints(np.expand_dims(pnts, axis=1), np.eye(3), None, None, np.eye(3))
    points3d = cv2.convertPointsToHomogeneous(pnts_und)
    rvec = np.zeros((1, 3), np.float32)
    tvec = np.zeros((1, 3), np.float32)
    return cv2.projectPoints(points3d.reshape(-1, 3), rvec, tvec, K, D)

#Репроекция точек
def reproj(pnts, K, D, R, T):
    tmp_pnts = []
    for pnt in pnts:
        pnt0 = np.dot(R, (pnt[0][0], pnt[0][1], 1.0)) + T
        tmp_pnts.append((pnt0[0]/pnt0[2], pnt0[1]/pnt0[2]))
    return distortPoints(tmp_pnts, K, D)

#Функция параллельной обработки
def parallel_processing(datas, processing_func, args, threads_number=-1):

    #Автоматическое определение оптимального числа потоков (если нужно)
    if threads_number <= 0:
        #Число потоков
        threads_number = cpu_count() + 1

    #Создание пула потоков
    with Pool(threads_number) as pool:

        #Запуск пула потоков
        pool_results = [pool.apply_async(processing_func, (copy.deepcopy(datas[args[i][0]]), copy.deepcopy(args[i][1]), i)) for i in np.arange(len(args))]
        #Закрытие пула потоков
        pool.close()
        #Ожидание завершения
        pool.join()
    
    #Возвращаем список результатов
    return pool_results

#Расчет реперных точек
def detect_points(image, mask, index):
    #Создание класса обнаружения реперных точек
    sift = cv2.SIFT_create(nfeatures=30000)
    #Расчет реперных точек слева
    kp, des = sift.detectAndCompute(image, mask)
    #Формирование списка точек
    coords = [p.pt for p in kp]
    #Параметры реперных точек
    return index, coords, des

#Расчет маски пересечения изобрадений слева и справа
def processing(argv):

    #Путь к конфигурационным данным
    path_config = argv[1]
    #Чтение конфигурационного файла
    with open(path_config) as fh:
        #Пропуск первой строки
        next(fh)
        #Чтение файла
        read_data = yaml.load(fh, Loader=yaml.FullLoader)

    #-------------- Извлечение конфигурационных параметров --------------#
    
    '''
    #Список наименований в пути
    names = path_config.split('/')
    #Путь к конфигурационному файлу
    path_data = ''
    #Цикл по наименованиям в пути
    for i in np.arange(len(names)-2):
        #Путь к конфигурационному файлу
        path_data += names[i] + '/'
    '''
    path_data = ''
    #Список имен камер
    cams = list(read_data['camera_streams'])
    #Список наименований файдов внутренних параметров
    intrinsics = read_data['intrinsics']
    #Наименование файла внешних параметров
    extrinsics = read_data['extrinsics']
    #Угол обзора камер
    view_ang = read_data['calibrate']['view_ang'] / 180 * math.pi
    #Положения камер
    positions = read_data['calibrate']['positions']
    #Точность позиционирования камер
    accuracy = read_data['calibrate']['accuracy']
    #Temp path images
    path_images = read_data['bired_eye_app']['temp_dir']

    #-------------- Индексация пар камер обработки --------------#
    
    #Условие калибровки всех пар
    if len(argv) == 3:
        #Признак обработки всех пар камер
        cam_ind = -1
    else:
        #Условие калибровки всех пар
        if len(argv) == 4:
            #Извлечение параметров пары камер
            pare_params = re.split("--|=|,", argv[3])
            #Условие корректной структуры сообщения
            if len(pare_params) != 4 or pare_params[1] != 'cams':
                #Признак неудачной калибровки
                return 1
            #Порядковый номер пары камер
            cam_ind = cams.index(pare_params[3])
        else:
            #Отладочная печать
            if DEBUG_PRINT: print('\033[93m' + 'Неверно указаны параметры пары камер...' + '\033[0m')
            #Признак неудачной калибровки
            return 1
        
    #-------------- Загрузка параметров камер --------------#
    
    #Список внутренних параметров камер
    ints = []
    #Цикл по камерам
    for cam in cams:
        #Загрузка файла с внутренними парамтерами камеры
        int_data = cv2.FileStorage(path_data + intrinsics[cam], cv2.FILE_STORAGE_READ)
        #Список внутренних параметров камер
        ints.append({'K': int_data.getNode("K").mat(), 'D': int_data.getNode("D").mat()})

    #Список внешних параметров
    exts = []
    #Загрузка файла с внутренними парамтерами камеры
    ext_data = cv2.FileStorage(path_data + extrinsics, cv2.FILE_STORAGE_READ)
    for cam in cams:
        exts.append({'R': ext_data.getNode("extrinsics").getNode(cam).getNode('R').mat(), 
                     'T': np.array([0, 0, 0], dtype='f')})

    #Признак обработки
    type_proc = argv[2]
    #Условие ресета
    if type_proc == '-r':

        #-------------- Формирование внешних параметров камер --------------#

        #Список матриц R
        Rt = []
        #Цикл по позициям
        for cam in positions:
            #Матрица поворота
            Rt.append(cv2.Rodrigues(np.array(positions[cam], dtype='f') / 180 * math.pi)[0])

        #-------------- Сохранение матриц {R,T} --------------#
        
        file_t = open(path_data + extrinsics, "wt")
        file_t.write('%YAML:1.0' + '\n')
        file_t.write('---' + '\n')
        file_t.write('extrinsics:' + '\n')

        #Цикл по камерам
        for i in np.arange(len(cams)):
            file_t.write('   ' + cams[i] + ':\n')
            file_t.write('      R: !!opencv-matrix' + '\n')
            file_t.write('         rows: 3' + '\n')
            file_t.write('         cols: 3' + '\n')
            file_t.write('         dt: f' + '\n')
            
            #Условие индекса сброса
            if i >= cam_ind:
                file_t.write('         data: ' + str(list(Rt[i].reshape(-1))) + '\n')
            else:
                file_t.write('         data: ' + str(list(exts[i]['R'].reshape(-1))) + '\n')
            #Вектор трансляции
            file_t.write('      T: ' + str([0., 0., 0.]) + '\n')
        
        #Закрытие файла
        file_t.close()

        #Признак удачного сброса параметров
        return 0

    #-------------- Расчет реперных пар точек --------------#

    #Пары соседних камер
    pares = []
    #Цикл по камерам
    for i in np.arange(len(cams)-1):
        #Признак значимости пары
        valid_pare = 1 if cam_ind == -1 or i+1 == cam_ind else 0
        #Пары соседних камер
        pares.append([i,i+1,valid_pare])  #1 - признак значимости пары

    #-------> Загрузка изображений

    #Отладочная печать
    if DEBUG_PRINT: print('Загрузка изображений...', end='\r')
    #Начало времени обработки
    start_time = time.time()

    #Список значимых индексов изображений
    image_inds = [0]*len(cams)
    #Заполнение вектора инжексов изображений
    for i in np.arange(len(pares)): image_inds[i] = 1; image_inds[i+1] = 1

    #Список изображений камер
    images = [None]*len(cams)
    #Цикл по камерам
    for i in np.arange(len(cams)):
        #Чтение изображения из локальной папки
        if image_inds[i] == 1: images[i] = cv2.cvtColor(cv2.imread(path_images + '/' + cams[i] + '.png'), cv2.COLOR_BGR2GRAY)
    
    #Конец времени обработки
    elapsed_time = time.time() - start_time
    #Отладочная печать
    if DEBUG_PRINT: print('Загрузка изображений...' + '\033[93m' + str(round(elapsed_time, 3)) + ' секунд ' + '\033[0m')

    #-------> Расчет реперных точек

    #Отладочная печать
    if DEBUG_PRINT: print('Расчет связанных реперных точек...')
    #Начало времени обработки
    start_time = time.time()

    #Список параметров
    params = []
    #Расчет маски пересечения изобрадений слева и справа
    mask0, mask1 = mask_process(images[0])
    #Цикл по парам камер
    for i in np.arange(len(pares)):
        #Индексы пары камер
        ind0, ind1, param = pares[i]
        #Условие незначимой пары
        if param == 0: continue
        #Список параметров слева
        params.append((ind0, mask0))
        #Список параметров справа
        params.append((ind1, mask1))

    #Функция параллельной обработки
    res_detect = parallel_processing(images, detect_points, params)
    #Сборка списка данных
    res_detect = [res.get() for res in res_detect]
    #Сортировка по порядковому номеру
    res_detect = sorted(res_detect, key = lambda x: x[0])

    #Порядковый номер
    j = 0
    #Список реперных точек для пар камер
    sift_pnts = [None]*len(pares)
    #Цикл по парам камер
    for i in np.arange(len(pares)):
        #Индексы пары камер
        ind0, ind1, param = pares[i]
        #Условие незначимой пары
        if param == 0: continue
        #Расчет реперных точек слева
        _, kp0, des0 = res_detect[j]; j += 1
        #Расчет реперных точек справа
        _, kp1, des1 = res_detect[j]; j += 1
        #Список реперных точек для пар камер
        sift_pnts[i] = ({'kp': kp0, 'des': des0}, {'kp': kp1, 'des': des1})

    #-------> Построение пар реперных точек

    #Признак алгоритма сопоставления точек
    FLANN_INDEX_KDTREE_SINGLE = 4 #FLANN_INDEX_HIERARCHICAL = 5
    #Индексы параметров
    index_params = dict(algorithm=FLANN_INDEX_KDTREE_SINGLE, trees=5)
    #Параметры поиска
    search_params = dict(checks=50)
    #Создание класса сопоставления точек
    flann = cv2.FlannBasedMatcher(index_params, search_params)

    #Список пар точек сопоставления
    lst_matches = [None]*len(pares)
    #Цикл по парам камер
    for i in np.arange(len(pares)):
        #Индексы пары камер
        ind0, ind1, param = pares[i]
        #Условие незначимой пары
        if param == 0: continue
        #Описание реперных точек
        des0, des1 = sift_pnts[i][0]['des'], sift_pnts[i][1]['des']
        #Сопоставление точек
        lst_matches[i] = flann.knnMatch(des0, des1, k=2)
        #Отладочная печать
        if DEBUG_PRINT: print('Количество пар точек ' + '(' + cams[ind0] + ',' + cams[ind1] + '):', len(lst_matches[i]))

        #Условие недостаточного количества пар точек
        if len(lst_matches[i]) < MIN_MATCHES:
            #Отладочная печать
            if DEBUG_PRINT: print('Количество пар точек меньше допустимого порога ' + '(' + cams[ind0] + ',' + cams[ind1] + '):', len(lst_matches[i]))
            #Признак невозможности выполнить калибровку пары камер
            pares[i][2] = 0
        
    #-------------- Выбор лучших пар точек --------------#
        
    #Список весов групп точек
    lst_weights = [None]*len(pares)
    #Список лучших групп точек
    lst_best_matches = [None]*len(pares)
    #Цикл по списку пар точек сопоставления
    for i in np.arange(len(lst_matches)):
        #Индексы пары камер
        ind0, ind1, param = pares[i]
        #Условие незначимой пары
        if param == 0: continue
        #Лучшие группы
        best_matches = []
        #Цикл по группам
        for m,n in lst_matches[i]:
            #Мера близости пары точек
            d0 = 0.6 * n.distance - m.distance
            #Степень подобия прямого и обратного проходов
            if d0 > 0: best_matches.append({'match':m, 'weight':(n.distance-m.distance)})
        #Фильтрация групп по общему углу сшивки
        best_matches = filter_points(sift_pnts[i][0]['kp'], sift_pnts[i][1]['kp'], best_matches, images[i].shape[1])
        #Сортировка групп по мере подобия (сначала самые правдоподобные)
        best_matches = sorted(best_matches, key = lambda x: x['weight'], reverse=True)[:MAX_MATCHES]

        #Условие недостаточного количества пар точек
        if len(best_matches) < MIN_MATCHES:
            #Отладочная печать
            if DEBUG_PRINT: print('Количество пар точек меньше допустимого ' + '(' + cams[ind0] + ',' + cams[ind1] + '):', len(best_matches))
            #Признак невозможности выполнить калибровку пары камер
            pares[i][2] = 0
            #Пропуск итерации
            continue

        #Минимальный/максимальный веса пар точек
        max_weight = best_matches[0]['weight']; min_weight = best_matches[-1]['weight']

        #Коэффициент масштабирования веса
        coeff = 1.0 / (max_weight - min_weight) if max_weight > min_weight else 1
        #Цикл по парам точек
        for match in best_matches:
            #Нормировка весового коэффициента
            match['weight'] = max(MIN_WEIGHT, (match['weight'] - min_weight) * coeff)

        #формирование значимого вектора весов пар точек
        lst_weights[i] = [match['weight'] for match in best_matches]
        #формирование значимого вектора подобия
        best_matches = [match['match'] for match in best_matches]

        #Список лучших групп точек
        lst_best_matches[i] = best_matches
        #Отладочная печать
        if DEBUG_PRINT: print('Количество значимых пар точек ' + '(' + cams[ind0] + ',' + cams[ind1] + '):', len(lst_best_matches[i]))

    #Конец времени обработки
    elapsed_time = time.time() - start_time
    #Отладочная печать
    if DEBUG_PRINT: print('Расчет связанных реперных точек...' + '\033[93m' + str(round(elapsed_time, 3)) + ' секунд ' + '\033[0m')

    '''
    #Цикл по парам камер
    for i in np.arange(len(pares)):
        #Индексы пары камер
        ind0, ind1, _ = pares[i]
        #Отображение точек сопоставления
        img3 = cv2.drawMatches(images[ind0], sift_pnts[i][0]['kp'], images[ind1], sift_pnts[i][1]['kp'], 
                            lst_best_matches[i], None, flags=cv2.DrawMatchesFlags_NOT_DRAW_SINGLE_POINTS)
        #Запись изобрадения в файл
        cv2.imwrite('/home/rubanov/calibrate/images/match' + str(i) + '.png', img3)
    '''
    
    #-------------- Оптимизация матрицы {R} --------------#

    #Размер области видимости
    viewSize = (images[0].shape[1], images[0].shape[0])
    #Матрица проекции
    viewK = MakeCameraMatrix(view_ang, viewSize)
    #Вектор дисторсии матрицы проекции
    viewD = np.array([0, 0, 0, 0], dtype='f')

    #Список точек пар сопоставлений
    pnts_matches = [None]*len(pares)
    #Цикл по парам камер
    for i in np.arange(len(pares)):
        #Индексы пары камер
        ind0, ind1, param = pares[i]
        #Условие отсутствия данных оптимизации
        if param == 0: continue
        #Список пар точек
        pnts0 = []; pnts1 = []
        #Описание реперных точек
        kp0, kp1 = sift_pnts[i][0]['kp'], sift_pnts[i][1]['kp']
        #Цикл по парам
        for match in lst_best_matches[i]:
            #Добавление точек слева
            pnts0.append(kp0[match.queryIdx])
            #Добавление точек справа
            pnts1.append(kp1[match.trainIdx])
        #Список точек пар сопоставлений
        pnts_matches[i] = (pnts0, pnts1)

    #Функция оптимизации
    def func_opt(x, pnts0, pnts1, weights, K0, D0, K1, D1):

        #Матрица поворота
        Rt = cv2.Rodrigues(x[:3])[0].reshape((3,3))
        #Матрица масштабирования
        Sc = x[3:].reshape((3,3))
        #Матрицы трансформации
        Rm = inv(np.dot(Rt,Sc))

        #Учет дисторсии
        pnts_und0 = cv2.undistortPoints(np.expand_dims(pnts0, axis=1), cameraMatrix=K0, distCoeffs=D0)
        #Репроекция
        pnts_reproj0, _ = reproj(pnts_und0, viewK, viewD, np.eye(3), np.zeros(3))

        #Учет дисторсии 
        pnts_und1 = cv2.undistortPoints(np.expand_dims(pnts1, axis=1), cameraMatrix=K1, distCoeffs=D1)
        #Репроекция
        pnts_reproj1, _ = reproj(pnts_und1, viewK, viewD, Rm, np.zeros(3))

        #Мера близости двух связаных групп точек 
        measure = measure_pnts(pnts_reproj0.reshape(-1,2), pnts_reproj1.reshape(-1,2), weights)
        #Мера близости двух связаных групп точек 
        return measure

    #Начало времени обработки
    start_time = time.time()
    #Отладочная печать
    if DEBUG_PRINT: print('Оптимизация матрицы {R}...', end='\r')

    #Вектор ошибок оптимизации
    measures = [None]*len(pares)
    #Цикл по группам точек пар камер
    for i in np.arange(len(pares)):
        #Индексы пары камер
        ind0, ind1, param = pares[i]
        #Условие отсутствия данных оптимизации
        if param == 0:
            #Вектор ошибок оптимизации
            measures[i] = -1
            #Пропуск итерации
            continue
        
        #Приращение матрицы трансформации
        dR = np.dot(exts[ind1]['R'], inv(exts[ind0]['R']))
        #Приращение вектора углов
        dV = cv2.Rodrigues(dR)[0].reshape(-1)
        #Приращение матрицы вращения
        dRt = cv2.Rodrigues(dV)[0].reshape((3,3))
        #Приращение матрицы масштабирования
        dS = np.dot(dR,inv(dRt))
        #Вектор оптимизации
        x_opt = np.concatenate((dV, dS.reshape(-1)))
        #Угловое приращение камер
        dpos = (np.array(positions[cams[ind1]], dtype='f') - np.array(positions[cams[ind0]], dtype='f')) / 180 * math.pi
        #Угол возвышения, рад.
        elevat = dpos[0]; delevat = accuracy[0] / 180 * math.pi
        #Угол азимута, рад.
        azimut = dpos[1]; dazimut = accuracy[1] / 180 * math.pi
        #Угол вращения, рад.
        roll = dpos[2]; droll = accuracy[2] / 180 * math.pi
        #Вектор ограничений
        x_bnds = np.array([( elevat-delevat, elevat+delevat),  #y
                           ( azimut-dazimut, azimut+dazimut),  #x 
                           ( roll-droll, roll+droll),          #angle
                           ( 0.95, 1.05), (-0.05, 0.05), (-0.05, 0.05), 
                           (-0.05, 0.05), ( 0.95, 1.05), (-0.05, 0.05),
                           (-0.05, 0.05), (-0.05, 0.05), ( 0.95, 1.05)], dtype='f').reshape((-1,2))
        #Обработка исключения
        try:
            #Реперные точки пары камер
            pnts0, pnts1 = pnts_matches[i]
            #Параметры функции оптимизации
            args = (pnts0, pnts1, lst_weights[i], ints[ind0]['K'], ints[ind0]['D'], ints[ind1]['K'], ints[ind1]['D'])
            #Оптимизация
            res = minimize(func_opt, x_opt, args=args, method='SLSQP', bounds=x_bnds, 
                           options={'ftol': 1e-3, 'disp': False, 'maxiter': 300})
            #Матрица поворота
            Ri = cv2.Rodrigues(res.x[:3])[0].reshape((3,3))
            #Матрица масштабирования
            Si = res.x[3:].reshape((3,3))
            #Локальная матрица трансформации
            Rl = np.dot(Ri,Si)
            #Полная матрица трансформации
            Rf = np.dot(Rl, exts[ind0]['R'])
            #Приращение полной матрицы трансформации
            dRf = np.dot(inv(exts[ind1]['R']),Rf)

            #Цикл по группам точек пар камер
            for k in np.arange(i, len(pares)):
                #Индексы пары камер
                _, ind1, _ = pares[k]
                #Преобразование матриц трасформации в последовательной цепочке
                exts[ind1]['R'] = np.dot(exts[ind1]['R'],dRf)
            
            #Вектор ошибок оптимизации
            measures[i] = round(res.fun, 3)
        except:
            #Вектор ошибок оптимизации
            measures[i] = -1
            #Отладочная печать
            if DEBUG_PRINT: print('\033[95m' + 'Неудачная оптимизация ' + '(' + cams[ind0] + ',' + cams[ind1] + '):' + '\033[0m')
            #Пропуск итерации
            continue
        
    #Конец времени обработки
    elapsed_time = time.time() - start_time
    #Отладочная печать
    if DEBUG_PRINT: print('Оптимизация матрицы {R}...' + '\033[93m' + str(round(elapsed_time, 3)) + ' секунд ' + '\033[0m')

    #Цикл по парам камер
    for i in np.arange(len(pares)):
        #Индексы пары камер
        ind0, ind1, param = pares[i]
        #Условие отсутствия данных оптимизации
        if param == 0: continue
        #Отладочная печать
        if DEBUG_PRINT: print('Ошибка оптимизации ' + '(' + cams[ind0] + ',' + cams[ind1] + '):', measures[i])
        #Отладочная печать
        if DEBUG_PRINT: print('Вектор вращения ' + '(' + cams[ind0] + ',' + cams[ind1] + '):', cv2.Rodrigues(exts[ind1]['R'])[0].reshape(-1))

    #-------------- Сохранение матрицы {R} --------------#
    
    file_t = open(path_data + extrinsics, "wt")
    file_t.write('%YAML:1.0' + '\n')
    file_t.write('---' + '\n')
    file_t.write('extrinsics:' + '\n')

    #Цикл по камерам
    for i in np.arange(len(cams)):
        file_t.write('   ' + cams[i] + ':\n')
        file_t.write('      R: !!opencv-matrix' + '\n')
        file_t.write('         rows: 3' + '\n')
        file_t.write('         cols: 3' + '\n')
        file_t.write('         dt: f' + '\n')
        file_t.write('         data: ' + str(list(exts[i]['R'].reshape(-1))) + '\n')
        file_t.write('      T: ' + str([0., 0., 0.]) + '\n')
    
    #Закрытие файла
    file_t.close()

    count = 0
    #Цикл по группам точек пар камер
    for pare in pares:
        #Условие отсутствия расчитанной пары
        count += pare[2]

    #Признак удачной калибровки
    return 1 if count == 0 else 0

#Основной метод обработки
res = processing(argv)
#Отладочная печать
if DEBUG_PRINT: print('Признак расчета всех заданных пар камер:', res)
#Возврат
sys.exit(res)

