const axios = require("axios");
const { Client } = require("pg");
require("dotenv").config({ path: ".env" });

const dbConfig = {
  user: process.env.DB_USER,
  host: process.env.DB_HOST,
  database: process.env.DB_NAME,
  password: process.env.DB_PASSWORD,
  port: process.env.DB_PORT,
};

const SENSORS_MOK = [
    {
    type: 2,
    detail: {
      id: "f043d4f5",
      ip: "192.168.1.103",
      status: 1,
      intensity: 2,
      coordinates: { x: 52.9301085852735, y: 36.00594436292393 },
      direction: 80,
    },
  },
  {
    type: 1,
    detail: {
      id: "3af742f2",
      ip: "192.168.2.133",
      status: 1,
      intensity: 2,
      coordinates: { x: 52.9346947767152, y: 36.99561937561036 },
      direction: 60,
    },
  },
  {
    type: 2,
    detail: {
      id: "fef093e-3",
      ip: "192.168.2.135",
      status: 1,
      intensity: 2,
      coordinates: { x: 52.92994942115238, y: 36.98993857012069 },
      direction: 10,
    },
  },
  {
    type: 2,
    detail: {
      id: "c27856ed",
      ip: "192.168.3.145",
      status: 0,
      intensity: 2,
      coordinates: { x: 52.94332615354757, y: 36.01798140359318 },
      direction: 10,
    },
  },
  {
    type: 1,
    detail: {
      id: "fd4047-3b",
      ip: "192.168.1.113",
      name: "Мотылёк",
      status: 1,
      direction: 10,
      intensity: 2,
      coordinates: { x: 52.694272, y: 35.07328 },
    },
  },
];

const API_URL = "http://localhost:5000/api/event";
const SEND_INTERVAL = 5000;
const statusList = [0, 1, 2];

const convertStatus = (input) => {
  const convertMap = new Map([
    [0, "Норма"],
    [1, "Тревога"],
    [2, "Авария"],
  ]);

  if (typeof input === "string") {
    for (const [key, value] of convertMap) {
      if (value === input) return key;
    }
  } else if (typeof input === "number") {
    return convertMap.get(input);
  }

  return 1;
};

const convertType = (input) => {
  const typeMap = new Map([
    [0, "Бинарный"],
    [1, "Стандартный"],
    [2, "Поворотный"],
  ]);

  if (typeof input === "string") {
    for (const [key, value] of typeMap) {
      if (value === input) return key;
    }
  } else if (typeof input === "number") {
    return typeMap.get(input);
  }

  throw new Error("Invalid input for conversion type");
};

async function getSensors() {
  const client = new Client(dbConfig);
  await client.connect();

  try {
    const res = await client.query("SELECT * FROM sensors");
    return res.rows;
  } catch (err) {
    console.error("Ошибка при получении данных из базы данных:", err);
    return [];
  } finally {
    await client.end();
  }
}

function changeSensorsStatus() {
  return statusList[Math.floor(Math.random() * statusList.length)];
}

function getRandomSensors(sensors) {
  const numberOfSensors = Math.floor(Math.random() * 3) + 1;
  const shuffledSensors = sensors.sort(() => 0.5 - Math.random());
  return shuffledSensors.slice(0, numberOfSensors);
}

async function sendData() {
  try {
    const dbSensors = await getSensors();
    let sensorsToSend = [];

    if (dbSensors.length === 0) {
      sensorsToSend = getRandomSensors(SENSORS_MOK).map((sensor) => ({
        ...sensor,
      }));
    } else {
      const dbSensorIds = new Set(dbSensors.map((sensor) => sensor.id));
      const uniqueMockSensors = SENSORS_MOK.filter(
        (mockSensor) => !dbSensorIds.has(mockSensor.id)
      );

      const allSensors = dbSensors.concat(uniqueMockSensors);
      sensorsToSend = getRandomSensors(allSensors).map((sensor) => ({
        type: convertType(sensor.type || "Стандартный"),
        detail: {
          id: sensor.id || sensor.detail.id,
          ip: sensor?.ip || sensor.detail?.ip || "12.12.12.12",
          name: sensor?.name || sensor.detail.name,
          intensity: sensor?.intensity || sensor.detail?.intensity || 1,
          direction: Math.floor(Math.random() * 1000) % 360,
          status: changeSensorsStatus(
            convertStatus(sensor?.status || sensor.detail.status)
          ),
          // status: 1,
          coordinates: sensor?.coordinates || sensor.detail.coordinates,
          createdAt: sensor?.createdAt || new Date(),
          updatedAt: sensor?.updatedAt || new Date(),
        },
      }));
    }
    const data = { events: sensorsToSend };
    await axios.post(API_URL, data, {
      headers: {
        "Content-Type": "application/json",
      },
    });

    console.log(`Событие отправлено: ${JSON.stringify(data.events)}`);
  } catch (error) {
    console.error("Ошибка в отправке события:", error.message);
  }
}

function startSendingData() {
  console.log(
    `Начало отправки данных каждые ${SEND_INTERVAL / 1000} секунд...`
  );
  setInterval(sendData, SEND_INTERVAL);
}

(async () => {
  startSendingData();
})();
